<?php

namespace App\Http\Controllers\Influencer;

use Pusher\Pusher;
use App\Models\User;
use App\Models\Hiring;
use App\Models\Influencer;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Models\GeneralSetting;
use App\Rules\FileTypeValidate;
use App\Models\HiringConversation;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class HiringController extends Controller
{
    public function index() {
        $this->pageTitle = 'All Hiring';
        return $this->filterHiring();
    }

    public function pending() {
        $this->pageTitle = 'Pending Hiring';
        return $this->filterHiring('pending');
    }

    public function inprogress() {
        $this->pageTitle = 'Processing Hiring';
        return $this->filterHiring('inprogress');
    }

    public function jobDone() {
        $this->pageTitle = 'Job Done Hiring';
        return $this->filterHiring('JobDone');
    }

    public function completed() {
        $this->pageTitle = 'Completed Hiring';
        return $this->filterHiring('completed');
    }

    public function reported() {
        $this->pageTitle = 'Reported Hiring';
        return $this->filterHiring('reported');
    }

    public function cancelled() {
        $this->pageTitle = 'Cancelled Hiring';
        return $this->filterHiring('cancelled');
    }

    public function detail($id) {
        $pageTitle = 'Hiring Detail';
        $hiring    = Hiring::where('influencer_id', authInfluencerId())->findOrFail($id);
        return view($this->activeTemplate . 'influencer.hiring.detail', compact('pageTitle', 'hiring'));
    }

    public function acceptStatus($id) {
        $influencer     = authInfluencer();
        $hiring         = Hiring::pending()->where('id', $id)->where('influencer_id', $influencer->id)->with('user')->first();
        if(!$hiring){

            $notify[] = ['warning', 'Before admin payment accepted and then your action'];
            return back()->withNotify($notify);
        }
        $hiring->status = 2;
        $hiring->save();

        $user    = $hiring->user;
        $general = gs();
        notify($user, 'HIRING_INPROGRESS', [
            'influencer'    => $influencer->username,
            'site_currency' => $general->cur_text,
            'amount'        => showAmount($hiring->amount),
            'hiring_no'     => $hiring->hiring_no,
            'title'         => $hiring->title,
        ]);
        $notify[] = ['success', 'Hiring status has now inprogress'];
        return back()->withNotify($notify);
    }

    public function jobDoneStatus($id) {
        $influencer     = authInfluencer();
        $hiring         = Hiring::inprogress()->where('id', $id)->where('influencer_id', $influencer->id)->with('user')->firstOrFail();
        $hiring->status = 3;
        $hiring->save();

        $user    = $hiring->user;
        $general = gs();
        notify($user, 'JOB_DONE_SUCCESSFULLY', [
            'influencer'    => $influencer->username,
            'site_currency' => $general->cur_text,
            'amount'        => showAmount($hiring->amount),
            'hiring_no'     => $hiring->hiring_no,
            'title'         => $hiring->title,
        ]);
        $notify[] = ['success', 'Job has been done successfully'];
        return back()->withNotify($notify);
    }

    public function cancelStatus($id) {

        $influencer     = authInfluencer();
        $hiring         = Hiring::where('id', $id)->where('influencer_id', $influencer->id)->with('user')->firstOrFail();
        $hiring->status = 5;
        $hiring->save();

        $user    = $hiring->user;
        $general = gs();

        $user->balance += $hiring->amount;
        $user->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $hiring->amount;
        $transaction->post_balance = $user->balance;
        $transaction->trx_type     = '+';
        $transaction->details      = showAmount($hiring->amount) . $general->cur_text . ' payment refunded for hiring cancellation';
        $transaction->trx          = getTrx();
        $transaction->remark       = 'hiring_payment';
        $transaction->save();

        notify($user, 'HIRING_CANCELLED', [
            'influencer'    => $influencer->username,
            'site_currency' => $general->cur_text,
            'amount'        => showAmount($hiring->amount),
            'post_balance'  => showAmount($user->balance),
            'hiring_no'     => $hiring->hiring_no,
            'title'         => $hiring->title,
        ]);
        $notify[] = ['success', 'Hiring has been cancelled successfully'];
        return back()->withNotify($notify);
    }

    protected function filterHiring($scope = null) {

        $influencerId = authInfluencerId();
        $hirings      = Hiring::query();

        if ($scope) {
            $hirings = $hirings->$scope();
        }
        $pageTitle = $this->pageTitle;

        $hirings = $hirings->where('influencer_id', $influencerId)->with('user')->latest()->paginate(getPaginate());
        return view($this->activeTemplate . 'influencer.hiring.index', compact('pageTitle', 'hirings'));
    }

    public function conversation($id) {
        $pageTitle = 'Conversation';
        $hiring  = Hiring::where('influencer_id', authInfluencerId())->with('hiringMessage')->findOrFail($id);
        $user  = User::where('id', $hiring->user_id)->first();
        $conversationMessage = HiringConversation::with('influencer', 'admin', 'user')->where('hiring_id', $hiring->id)->latest()->get();
        return view($this->activeTemplate . 'influencer.hiring.conversation', compact('pageTitle', 'conversationMessage', 'user', 'hiring'));
    }

    public function conversationStore(Request $request) {


        $hiring = Hiring::where('influencer_id', authInfluencerId())->find($request->hiring_id);
        $influencer = authInfluencer();

        if (!$hiring) {
            return response()->json(['error' => 'Hiring id not found.']);
        }

        $validator = Validator::make($request->all(), [
            'message'       => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()]);
        }

        $user = User::active()->find($hiring->user_id);

        if (!$user) {
            return response()->json(['error' => 'Influencer is banned from admin.']);
        }
        $general = GeneralSetting::first();

        $message                = new HiringConversation();
        $message->hiring_id     = $hiring->id;
        $message->user_id       = $user->id;
        $message->influencer_id = authInfluencerId();
        $message->sender        = 'influencer';
        $message->message       = $request->message;
        $message->save();

        $senderImagePath = getImage(getFilePath('influencerProfile') . '/' . ($influencer->image ?? ''), getFileSize('influencerProfile'));
        $sendTime = date('h:i a',strtotime($message->created_at));

        $orderData = [
            'admin_id' => 1,
            'user_id'=>$hiring->user_id
        ];

        foreach($orderData as $item)
        {
            $options = [
                'cluster' => $general->pusher_credential->app_cluster,
                'encrypted' => $general->pusher_credential->useTLS
            ];

            $pusher = new Pusher(
                $general->pusher_credential->app_key,
                $general->pusher_credential->app_secret,
                $general->pusher_credential->app_id,
                $options
            );

            $data = [
                'id' => $message->id,
                'message' => $message->message,
                'sender' => $message->influencer_id,
                'senderName' => $influencer->username,
                'receiver' => $item,
                'created_at' => $message->created_at,
                'senderImag' => $senderImagePath,
                'sendTime' => $sendTime,
            ];

            $event_name = '' .$item. '';
            $pusher->trigger($event_name, "App\\Events\\HiringChat", $data);

        }

        return response()->json($data);
    }

}
