<?php

namespace App\Http\Controllers\Influencer;

use App\Models\Category;
use App\Models\SocialLink;
use Illuminate\Http\Request;
use App\Rules\FileTypeValidate;
use App\Models\InfluencerEducation;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use App\Models\InfluencerQualification;
use Illuminate\Validation\Rules\Password;

class ProfileController extends Controller
{
    public function profile()
    {
        $pageTitle = "Profile Setting";
        $user = authInfluencer();
        $languageData = config('languages');
        $countries = json_decode(file_get_contents(resource_path('views/includes/country.json')));
        $categories =  Category::active()->get();
        return view($this->activeTemplate. 'influencer.profile_setting', compact('pageTitle','user','languageData','countries','categories'));
    }

    public function submitProfile(Request $request)
    {
        $request->validate([
            'firstname' => 'required|string',
            'lastname' => 'required|string',
            'profession' => 'nullable|max:40|string',
            'summary'    => 'nullable|string',
            'image'        => ['nullable','image', new FileTypeValidate(['jpeg', 'jpg', 'png'])],
        ],[
            'firstname.required'=>'First name field is required',
            'lastname.required'=>'Last name field is required'
        ]);

        $user = authInfluencer();

        $user->firstname = $request->firstname;
        $user->lastname = $request->lastname;

        $user->address = [
            'address' => $request->address,
            'state' => $request->state,
            'zip' => $request->zip,
            'country' => @$user->address->country,
            'city' => $request->city,
        ];

        $user->profession = $request->profession;
        $user->summary    = nl2br($request->summary);

        if ($request->category) {
            $categoriesId = $request->category;
            $user->categories()->sync($categoriesId);
        }

        if ($request->hasFile('image'))
        {
            $path = getFilePath('influencerProfile');
            fileManager()->removeFile($path.'/'.$user->image);
            $directory = $user->username."/". $user->id;
            $path = getFilePath('influencerProfile').'/'.$directory;
            $filename = $directory.'/'.fileUploader($request->image, $path, getFileSize('influencerProfile'));
            $user->image = $filename;
        }

        $user->save();
        $notify[] = ['success', 'Profile has been updated successfully'];
        return back()->withNotify($notify);
    }

    public function changePassword()
    {
        $pageTitle = 'Change Password';
        return view($this->activeTemplate . 'influencer.password', compact('pageTitle'));
    }

    public function submitPassword(Request $request)
    {

        $passwordValidation = Password::min(6);
        $general = gs();
        if ($general->secure_password) {
            $passwordValidation = $passwordValidation->mixedCase()->numbers()->symbols()->uncompromised();
        }

        $this->validate($request, [
            'current_password' => 'required',
            'password' => ['required','confirmed',$passwordValidation]
        ]);

        $user = authInfluencer();
        if (Hash::check($request->current_password, $user->password)) {
            $password = Hash::make($request->password);
            $user->password = $password;
            $user->save();
            $notify[] = ['success', 'Password changes successfully'];
            return back()->withNotify($notify);
        } else {
            $notify[] = ['error', 'The password doesn\'t match!'];
            return back()->withNotify($notify);
        }
    }

    public function submitSkill(Request $request) {

        $request->validate([
            "skills"   => "nullable|array",
            "skills.*" => "required|string",
        ]);

        $influencer         = authInfluencer();
        $influencer->skills = $request->skills;
        $influencer->save();

        $notify[] = ['success', 'Skill added successfully'];
        return back()->withNotify($notify);
    }

    public function addLanguage(Request $request)
    {
        $request->validate([
            'name'      => 'required|string|max:40',
            'listening' => 'required|in:Basic,Medium,Fluent',
            'speaking'  => 'required|in:Basic,Medium,Fluent',
            'writing'   => 'required|in:Basic,Medium,Fluent',
        ]);

        $influencer   = authInfluencer();
        $oldLanguages = authInfluencer()->languages ?? [];

        if (!is_array($oldLanguages)) {
            $oldLanguages = [];
        }

        $addedLanguages = array_keys($oldLanguages);

        if (in_array(strtolower($request->name), array_map('strtolower', $addedLanguages))) {
            $notify[] = ['error', $request->name . ' already added'];
            return back()->withNotify($notify);
        }

        $newLanguage[$request->name] = [
            'listening' => $request->listening,
            'speaking'  => $request->speaking,
            'writing'   => $request->writing,
        ];

        $languages = array_merge($oldLanguages, $newLanguage);

        $influencer->languages = $languages;
        $influencer->save();

        $notify[] = ['success', 'Language added successfully'];
        return back()->withNotify($notify);
    }

    public function removeLanguage($language) {
        $influencer     = authInfluencer();
        $oldLanguages   = $influencer->languages ?? [];
        $addedLanguages = array_keys($oldLanguages);

        if (in_array($language, $addedLanguages)) {
            unset($oldLanguages[$language]);
        }

        $influencer->languages = $oldLanguages;
        $influencer->save();

        $notify[] = ['success', 'Language removed successfully'];
        return back()->withNotify($notify);
    }

    public function addSocialLink(Request $request, $id = 0) {
        $request->validate([
            'social_icon' => 'required',
            'url'         => 'required',
            'followers'   => 'required|string|max:40',
        ]);

        $influencerId = authInfluencerId();

        if ($id) {
            $social       = SocialLink::where('influencer_id', $influencerId)->findOrFail($id);
            $notification = 'Social link updated successfully';
        } else {
            $social                = new SocialLink();
            $social->influencer_id = $influencerId;
            $notification          = 'Social link added successfully';
        }

        $social->social_icon = $request->social_icon;
        $social->url         = $request->url;
        $social->followers   = $request->followers;
        $social->save();

        $notify[] = ['success', $notification];
        return back()->withNotify($notify);
    }

    public function removeSocialLink($id) {

        $influencerId = authInfluencerId();
        SocialLink::where('influencer_id', $influencerId)->findOrFail($id)->delete();
        $notify[] = ['success', 'Social link removed successfully'];
        return back()->withNotify($notify);
    }

    public function addEducation(Request $request, $id = 0) {

        $request->validate([
            'country'    => 'required|string',
            'institute'  => 'required|string',
            'degree'     => 'required|string',
            'start_year' => 'required|date_format:Y',
            'end_year'   => 'required|date_format:Y|after_or_equal:start_year',
        ]);

        $influencerId = authInfluencerId();

        if ($id) {
            $education    = InfluencerEducation::where('influencer_id', $influencerId)->findOrFail($id);
            $notification = 'Education updated successfully';
        } else {
            $education                = new InfluencerEducation();
            $education->influencer_id = $influencerId;
            $notification             = 'Education added successfully';
        }

        $education->country    = $request->country;
        $education->institute  = $request->institute;
        $education->degree     = $request->degree;
        $education->start_year = $request->start_year;
        $education->end_year   = $request->end_year;
        $education->save();

        $notify[] = ['success', $notification];
        return back()->withNotify($notify);
    }

    public function removeEducation($id) {
        $influencerId = authInfluencerId();
        InfluencerEducation::where('influencer_id', $influencerId)->where('id', $id)->delete();
        $notify[] = ['success', 'Education remove successfully'];
        return back()->withNotify($notify);
    }

    public function addQualification(Request $request, $id = 0) {
        $request->validate([
            'certificate'  => 'required|string',
            'organization' => 'required|string',
            'summary'      => 'nullable|string',
            'year'         => 'required|date_format:Y',
        ]);

        $influencerId = authInfluencerId();

        if ($id) {
            $education    = InfluencerQualification::where('influencer_id', $influencerId)->findOrFail($id);
            $notification = 'Qualification updated successfully';
        } else {
            $education                = new InfluencerQualification();
            $education->influencer_id = $influencerId;
            $notification             = 'Qualification added successfully';
        }

        $education->certificate  = $request->certificate;
        $education->organization = $request->organization;
        $education->summary      = $request->summary;
        $education->year         = $request->year;
        $education->save();

        $notify[] = ['success', $notification];
        return back()->withNotify($notify);
    }

    public function removeQualification($id) {
        $influencerId = authInfluencerId();
        InfluencerQualification::where('influencer_id', $influencerId)->where('id', $id)->delete();
        $notify[] = ['success', 'Qualification remove successfully'];
        return back()->withNotify($notify);
    }







}
