<?php

namespace App\Http\Controllers\Influencer;

use App\Models\Tag;
use App\Models\Service;
use App\Models\Category;
use Illuminate\Http\Request;
use App\Rules\FileTypeValidate;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class ServiceController extends Controller
{
    public function index(){

        $pageTitle = 'All Services';
        $services  = $this->serviceData();
        return view($this->activeTemplate .'influencer.services.index', compact('pageTitle', 'services'));
    }

    public function create() {

        $pageTitle = "Create Service";
        $categories = Category::active()->latest()->get();
        return view($this->activeTemplate .'influencer.services.create', compact('pageTitle','categories'));
    }

    public function store(Request $request){

        $this->validation($request->all())->validate();
        $purifier = new \HTMLPurifier();
        $influencerId = authInfluencerId();

        $service = new Service();
        $service->influencer_id = $influencerId;
        $service->category_id =  $request->category_id;
        $service->title =  $request->title;
        $service->price =  $request->price;
        $service->key_points = $request->key_points;
        $service->description = $purifier->purify($request->description);
        $service->status = gs()->service_approve == 1 ?? 0;

        if ($request->hasFile('image')) {
            try {
                $service->image = fileUploader($request->image, getFilePath('serviceImage'), getFileSize('serviceImage'),$old =null);
            } catch (\Exception$exp) {
                $notify[] = ['error', 'Couldn\'t upload your image'];
                return back()->withNotify($notify);
            }

        }

        $service->save();
        $this->insertTag($service);

        $notify[] = ['success', 'Service Created'];
        return back()->withNotify($notify);
    }

    public function edit($id){
        $pageTitle = "Update Service";
        $categories = Category::active()->latest()->get();
        $service = Service::findOrFail($id);
        return view($this->activeTemplate.'influencer.services.edit',compact('pageTitle','categories','service'));
    }

    public function update(Request $request,$id){

        $this->validation($request->all(),$id)->validate();
        $purifier = new \HTMLPurifier();
        $influencerId = authInfluencerId();

        $service = Service::findOrFail($id);
        $service->influencer_id = $influencerId;
        $service->category_id =  $request->category_id;
        $service->title =  $request->title;
        $service->price =  $request->price;
        $service->key_points = $request->key_points;
        $service->description = $purifier->purify($request->description);

        if ($request->hasFile('image')) {
            try {
                $old = $service->image;
                $service->image = fileUploader($request->image, getFilePath('serviceImage'), getFileSize('serviceImage'),$old);
            } catch (\Exception$exp) {
                $notify[] = ['error', 'Couldn\'t upload your image'];
                return back()->withNotify($notify);
            }

        }

        $service->save();
        $this->insertTag($service,$id);
        $notify[] = ['success', 'Service Updated'];
        return back()->withNotify($notify);

    }

    protected function insertTag($service, $id=null) {
        $request = request();
        foreach ($request->tags as $tag) {
            $tagExist = Tag::where('name', $tag)->first();

            if ($tagExist) {
                $tagId[] = $tagExist->id;
            } else {
                $newTag       = new Tag();
                $newTag->name = $tag;
                $newTag->save();
                $tagId[] = $newTag->id;
            }

        }

        if ($id) {
            $service->tags()->sync($tagId);
        } else {
            $service->tags()->attach($tagId);
        }

    }

    protected function validation(array $data, $id=null) {

        $imageValidation = !$id ? 'required' : 'nullable';

        $validate = Validator::make($data, [
            'category_id'  => 'required|integer|exists:categories,id',
            'title'        => 'required|string|max:255',
            'description'  => 'nullable|string',
            'price'        => 'required|numeric|gte:0',
            'tags'         => 'required|array|min:1',
            'key_points'   => 'required|array|min:1',
            'key_points.*' => 'required|string|max:255',
            'image'        => [$imageValidation, 'image', new FileTypeValidate(['jpeg', 'jpg', 'png'])],
            'images'       => 'nullable|array',
            'images.*'     => ['nullable', 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ], [
            'key_points.*.required' => 'Key points is required',
        ]);

        return $validate;
    }

     protected function serviceData($scope = null) {
        if ($scope) {
            $services = Service::$scope();
        } else {
            $services = Service::query();
        }

        $services = $services->where('influencer_id', authInfluencerId());

        return $services->with('category')->latest()->paginate(getPaginate());
    }

}
