<?php

namespace App\Http\Controllers\User;

use Pusher\Pusher;
use App\Models\Order;
use App\Models\Service;
use App\Models\Influencer;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Models\GeneralSetting;
use App\Models\GatewayCurrency;
use App\Models\AdminNotification;
use App\Models\OrderConversation;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;

class OrderController extends Controller
{
    public function index() {
        $this->pageTitle = 'All Orders';
        return $this->filterOrder();
    }

    public function pending() {
        $this->pageTitle = 'Pending Orders';
        return $this->filterOrder('pending');
    }

    public function inprogress() {
        $this->pageTitle = 'Processing Orders';
        return $this->filterOrder('inprogress');
    }

    public function jobDone() {
        $this->pageTitle = 'Job Done Orders';
        return $this->filterOrder('JobDone');
    }

    public function completed() {
        $this->pageTitle = 'Completed Orders';
        return $this->filterOrder('completed');
    }

    public function reported() {
        $this->pageTitle = 'Reported Orders';
        return $this->filterOrder('reported');
    }

    public function cancelled() {
        $this->pageTitle = 'Cancelled Orders';
        return $this->filterOrder('cancelled');
    }

    protected function filterOrder($scope = null) {

        $user = auth()->user();
        $orders       = Order::query();

        if ($scope) {
            $orders = $orders->$scope();
        }

        $orders = $orders->where('user_id', $user->id)->with('influencer')->latest()->paginate(getPaginate());

        $pageTitle = $this->pageTitle;
        return view($this->activeTemplate . 'user.orders.index', compact('pageTitle', 'orders'));
    }


    public function detail($id)
    {
        $pageTitle = 'Order Detail';
        $order     = Order::where('user_id', auth()->id())->with('influencer', 'service')->findOrFail($id);
        return view($this->activeTemplate . 'user.orders.detail', compact('pageTitle', 'order'));
    }

    public function reportStatus(Request $request, $id)
    {
        $request->validate([
            'reason' => 'required|string',
        ]);

        $user          = auth()->user();
        $order         = Order::where('id', $id)->where('user_id', $user->id)->with('influencer')->firstOrFail();
        $order->status = 4;
        $order->reason = $request->reason;
        $order->save();

        $influencer = $order->influencer;


        notify($influencer, 'ORDER_REPORTED', [
            'username'      => $user->username,
            'title'         => $order->title,
            'amount'        => showAmount($order->amount),
            'order_no'      => $order->order_no,
            'reason'        => $order->reason,
        ]);

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = 'This order is reported by ' . $user->username;
        $adminNotification->click_url = urlPath('admin.order.detail', $order->id);
        $adminNotification->save();

        $notify[] = ['success', 'You report submitted successfully. Admin will take action immediately.'];
        return back()->withNotify($notify);
    }

    public function completeStatus($id)
    {
        $user          = auth()->user();
        $order         = Order::JobDone()->where('id', $id)->where('user_id', $user->id)->with('influencer')->firstOrFail();
        $order->status = 1;
        $order->save();

        $influencer = $order->influencer;

        $influencer->balance += $order->amount;
        $influencer->increment('completed_order');
        $influencer->save();

        notify($influencer, 'ORDER_COMPLETED_INFLUENCER', [
            'title'         => $order->title,
            'amount'        => showAmount($order->amount),
            'order_no'      => $order->order_no,
        ]);

        $transaction                = new Transaction();
        $transaction->influencer_id = $influencer->id;
        $transaction->amount        = $order->amount;
        $transaction->post_balance  = $influencer->balance;
        $transaction->trx_type      = '+';
        $transaction->details       = 'Payment received for completing a new service order';
        $transaction->trx           = getTrx();
        $transaction->remark        = 'order_payment';
        $transaction->save();

        $notify[] = ['success', 'Order completed successfully'];
        return back()->withNotify($notify);
    }

    public function orderRequestForm($id)
    {
        $service = Service::approved()
            ->whereHas('influencer', function ($influencer) {
                return $influencer->active();
            })->where('id', $id)->firstOrFail();

        $pageTitle  = 'Order Request';
        $influencer = $service->influencer;

        return view($this->activeTemplate . 'user.orders.order_request', compact('pageTitle', 'service', 'influencer'));
    }

    public function orderServicePaymentMethod(Request $request)
    {
        $this->validation($request);

        Session::put('order_data', [
            'influencer_id' =>$request->influencer_id,
            'service_id' =>$request->service_id,
            'title' =>$request->title,
            'delivery_date' => $request->delivery_date,
            'price' => $request->price,
            'description' => $request->description,
        ]);

        $hiringData = Session::get('order_data');
        if (!$hiringData) {
            $notify[] = ['error', 'Session Invalidate'];
            return redirect()->route('user.home')->withNotify($notify);
        }
        $amount = $hiringData['price'];

        $gatewayCurrency = GatewayCurrency::whereHas('method', function ($gate) {
            $gate->where('status', 1);
        })->with('method')->orderby('method_code')->get();
        $pageTitle = 'Payment Methods';

        return view($this->activeTemplate . 'user.payment.order_payment', compact('gatewayCurrency', 'pageTitle','amount'));
    }

    public function saveOrderData($orderData, $paymentStatus=null)
    {
        $user = auth()->user();
        $order = new Order();
        $order->user_id = $user->id;
        $order->influencer_id = $orderData['influencer_id'];
        $order->service_id = $orderData['service_id'];
        $order->title = $orderData['title'];
        $order->delivery_date = $orderData['delivery_date'];
        $order->amount  = $orderData['price'];
        $order->description = $orderData['description'];
        $order->status = 0;
        $order->payment_status = $paymentStatus;
        $order->order_no = getTrx();
        $order->save();

        return $order;
    }

    public function payViaWallet($order)
    {
        $user = auth()->user();
        $user->balance -= $order->amount;
        $user->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $order->amount;
        $transaction->post_balance = $user->balance;
        $transaction->trx_type     = '-';
        $transaction->trx          = getTrx();
        $transaction->details      = 'Balance deducted for ordering a service.';
        $transaction->remark       = 'order_payment';
        $transaction->save();

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = 'A new order placed by ' . $user->username;
        $adminNotification->click_url =  urlPath('admin.order.detail', $order->id);
        $adminNotification->save();

        notify($user, 'ORDER_PLACED', [
            'username'      => $user->username,
            'title'         => $order->title,
            'amount'        => showAmount($order->amount),
            'order_no'      => $order->order_no,
        ]);
    }

    protected function validation($request)
    {
        $request->validate([
            'title'         => 'required|string|max:255',
            'delivery_date' => 'required|date_format:Y-m-d|after:yesterday',
            'price'        => 'required|numeric|gt:0',
            'description'   => 'required|string',
        ]);
    }

    public function conversation($id)
    {
        $pageTitle           = 'Order Conversation';
        $user = auth()->user();
        $order  = Order::where('user_id', $user->id)->with('orderMessage')->findOrFail($id);
        $influencer  = Influencer::where('id', $order->influencer_id)->first();
        $conversationMessage = OrderConversation::with('influencer', 'admin', 'user')->where('order_id', $order->id)->latest()->get();
        return view($this->activeTemplate . 'user.orders.conversation', compact('pageTitle', 'conversationMessage', 'influencer', 'order'));
    }

    public function conversationStore(Request $request) {

        $user = auth()->user();
        $order = Order::where('user_id', $user->id)->find($request->order_id);

        if (!$order) {
            return response()->json(['error' => 'Order not found.']);
        }

        $validator = Validator::make($request->all(), [
            'message'       => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()]);
        }


        $general = GeneralSetting::first();

        $message                = new OrderConversation();
        $message->order_id      = $order->id;
        $message->user_id       = $user->id;
        $message->influencer_id =  $order->influencer_id;
        $message->sender        = 'client';
        $message->message       = $request->message;
        $message->save();


        $senderImagePath = getImage(getFilePath('userProfile') . '/' . ($user->image ?? ''), getFileSize('userProfile'));
        $sendTime = date('h:i a',strtotime($message->created_at));

        $orderData = [
            'admin_id' => 1,
            'influencer_id'=>$order->influencer_id
        ];


        foreach($orderData as $item)
        {

             $options = [
            'cluster' => $general->pusher_credential->app_cluster,
            'encrypted' => $general->pusher_credential->useTLS
            ];


            $pusher = new Pusher(
                $general->pusher_credential->app_key,
                $general->pusher_credential->app_secret,
                $general->pusher_credential->app_id,
                $options
            );

            $data = [
                'id' => $message->id,
                'message' => $message->message,
                'sender' => $message->user_id,
                'senderName' => $user->username,
                'receiver' =>   $item,
                'created_at' => $message->created_at,
                'senderImag' => $senderImagePath,
                'sendTime' => $sendTime,
            ];

            $event_name = '' .  $item . '';
            $pusher->trigger($event_name, "App\\Events\\HiringChat", $data);

        }

        return response()->json($data);

    }


}
